<?php

namespace Tss;

use FilesystemIterator;
use Illuminate\Support\Facades\Route;
use ReflectionClass;
use ReflectionEnum;
use Illuminate\Support\Str;

use Illuminate\Routing\Route as LaravelRoute;
use ReflectionMethod;
use Symfony\Component\Finder\Iterator\RecursiveDirectoryIterator;

use Illuminate\Routing\RouteCollection;

class Generator
{
    /**
     * The TS import location from which the 'apiRequest' function is imported.
     */
    private ?string $importApiRequestFrom = null;

    /**
     * The types that have been added to the generator, keyed by the type name and
     * the value being the trimmed TS output code.
     */
    private array $types = [];

    /**
     * The code snippets that have been added to the generator
     */
    private array $codes = [];

    /**
     * The routes
     */
    private array $routes = [];

    /**
     * The complete generated TS output code, cached after the first call to output().
     */
    private ?string $cached_output = null;

    /**
     * Add a type to the generator. Convenience function around addClass() and addEnum().
     */
    public function addType(string $type)
    {
        if (enum_exists($type)) {
            $this->addEnum($type);
        } else {
            $this->addClass($type);
        }
    }

    /**
     * Add an enum to the generator, with an optional override name.
     */
    public function addEnum(string $enum, string $overrideName = null)
    {
        // Use the basename of the class if no override is provided
        $name = $overrideName ?? (new ReflectionEnum($enum))->getShortName();
        $cases = [];
        foreach ($enum::cases() as $c) {
            $cases[] = $c->name;
        }
        $output = "export type {$name} = '" . implode("' | '", $cases) . "';";
        $this->types[$name] = $output;
    }

    /**
     * Add a class-based const/enum equivalent, with an optional override name.
     */
    public function addClassConstantEnum(string $class, string $overrideName = null)
    {
        $name = $overrideName ?? (new ReflectionClass($class))->getShortName();
        $keys = [];
        foreach ((new ReflectionClass($class))->getConstants() as $k => $v) {
            if ($k !== $v) {
                throw new TypeScriptGeneratorException("Can only add class-based contants enums where the keys match the values. See '$k' in $name");
            }
            $keys[] = $k;
        }
        sort($keys);
        $output = "export type {$name} =";
        foreach ($keys as $k) {
            $output .= "\n  | '$k'";
        }
        $output .= ";";
        $this->types[$name] = $output;
    }

    private function cleanupCode(string $code, bool $indent = true)
    {
        $code = trim($code);
        $code = preg_replace('/^\s*/m', '  ', $code);
        // $code = preg_replace('/^\s*\* /m', '', $code);
        // $code = preg_replace('/^/m', '  ', $code);
        return ($indent ? '  ' : '') . trim($code) . ($indent ? "\n" : '');
    }

    public function importApiRequestFrom(string $import)
    {
        $this->importApiRequestFrom = $import;
    }

    /**
     * Add a class to the generator.
     */
    public function addClass(string $class)
    {
        throw new TypeScriptGeneratorException("No support for adding class: $class");
    }

    /**
     * Add a class to the generator.
     */
    public function addCode(string ...$codes)
    {
        foreach ($codes as $c) {
            $this->codes[] = $this->cleanupCode($c, false);
        }
    }

    private function isSimpleType(string $type): bool
    {
        return (bool) preg_match('/^[a-zA-Z][a-zA-Z0-9\<\>\[\]_]*$/', $type);
    }

    /**
     * Loop through the routes of the application, generating TS wrappers for the ones marked as such
     * with TS attribute annotations.
     */
    public function addRoutes($routes)
    {
        /**
         * Collect all required metadata into a routes array, before we start
         * investigating each individual route:
         */
        $metaroutes = [];
        foreach ($routes as $r) {
            // Read TS attributes for the controller action method
            $cb = Str::parseCallback($r->action['uses']);

            $ref = new ReflectionMethod($cb[0], $cb[1]);
            $attr = $ref->getAttributes(TypeScript::class);
            $ts = [];
            $code = [];


            $rx = preg_match_all('/(?<whole>\{(?<arg>[A-Za-z_]+)(?<suffix>:[A-Za-z_0-9]+)?\})/', $r->uri, $matches, PREG_OFFSET_CAPTURE | PREG_SET_ORDER);
            if (0 === $rx) {
                $jspath = "'/" . $r->uri . "'";
                $params = [];
                // No parameters in this URL
            } else {
                $jspath = $r->uri;
                $params = [];
                // Reverse the matches so that we can replace the parameters in the correct order
                foreach (array_reverse($matches) as $m) {
                    $jspath = substr($jspath, 0, $m['whole'][1]) . '${encodeURIComponent(' . $m['arg'][0] . ')}' . substr($jspath, $m['whole'][1] + strlen($m['whole'][0]));
                    $pn = $m['arg'][0];

                    $type = (isset($r->wheres[$pn])
                        && $r->wheres[$pn] === '[0-9]+')
                        ? 'number'
                        : 'string';

                    $params[] = [
                        'name' => $pn,
                        'type' => $type,
                    ];
                }
                $jspath = '`/' . $jspath . '`';
            }

            if ($attr) {
                foreach ($attr as $att) {
                    foreach ($att->getArguments() as $k => $v) {
                        switch ($k) {
                            case 'name':
                            case 'input':
                            case 'output':
                                $ts[$k] = $v;
                                break;
                            case 'code':
                                $code[] = $v;
                                break;
                            default:
                                throw new TypeScriptGeneratorException("Unknown TS attribute: $k");
                        }
                    }
                }
            } else {
                // No TS attributes, skip this route
                continue;
            }

            // Skip HEAD methods, as they are not supported by the TS API client
            $methods = collect($r->methods)
                ->filter(fn ($m) => $m !== 'HEAD')
                ->values()
                ->toArray();

            // Check that the route has exactly one HTTP method, but we only
            // need to do this after checking the endpoint should be part of
            // the TS export at all.
            if (count($methods) !== 1) {
                throw new TypeScriptGeneratorException("Route {$r->getName()} has more than one HTTP method: " . implode(', ', $methods));
                // If we need to support this in the future, we can prefix/suffix each route with the http method
            }

            $this->addCode(...$code);

            $name =  $ts['name'] ?? $r->getName();

            $input_name = null;
            $output_name = null;

            // Check input
            if (isset($ts['input'])) {
                $input_name = ucfirst($name) . 'Input';
                $this->types[$input_name] = $this->isSimpleType($ts['input'])
                    ? "export type $input_name = {$ts['input']};"
                    : "export interface $input_name {\n" . $this->cleanupCode($ts['input']) . "}";
            }
            if (isset($ts['output'])) {
                $output_name = ucfirst($name) . 'Result';
                $this->types[$output_name] = $this->isSimpleType($ts['output'])
                    ? "export type $output_name = {$ts['output']};"
                    : "export interface $output_name {\n" . $this->cleanupCode($ts['output']) . "}";
            }

            $metaroutes[] = [
                'route_name' => $r->getName(),
                'controller_class' => $cb[0],
                'controller_method' => $cb[1],
                'method' => strtolower($methods[0]),
                'uri' => '/' . $r->uri,
                'ts' => $ts,
                'input' => $input_name,
                'output' => $output_name,
                'api_call_function_name' => $name,
                'jspath' => $jspath,
                'params' => $params,
            ];
        }

        /**
         * Loop through each of the parsed routes in the collection, creating a TS wrapper for each of them.
         */
        $calls = [];
        foreach ($metaroutes as $r) {
            $parts = [];
            $parts[] = 'export const ' . $r['api_call_function_name'] . ' = async (';

            $args = [];

            if (count($r['params'])) {
                foreach (array_reverse($r['params']) as $prm) {
                    $args[] = "{$prm['name']}: {$prm['type']}";
                }
            }

            if ($r['input']) {
                $args[] = 'data: ' . $r['input'];
            }
            $parts[] = implode(', ', $args);

            $parts[] = ') => apiRequest';
            $parts[] = '<' . ($r['input'] ?? 'undefined') . ', ' . ($r['output'] ?? 'void') . '>';

            $parts[] = "('{$r['method']}', {$r['jspath']}, ";
            $parts[] = isset($r['ts']['input']) ? 'data' : 'undefined';
            $parts[] = ');';

            $calls[] = implode('', $parts);
        }


        /**
         * Loop through the parsed routes and read attributes etc. for each of them
         */
        $this->routes = $calls;
    }


    private function generateOutput()
    {
        $this->cached_output = '';

        if ($this->importApiRequestFrom) {
            $this->cached_output = "import { apiRequest } from '" . $this->importApiRequestFrom . "';\n\n";
        }

        // Add the types
        foreach ($this->types as $type) {
            $this->cached_output .= $type . "\n\n";
        }

        // Add the code snippets
        foreach ($this->codes as $code) {
            $this->cached_output .= $code . "\n\n";
        }

        foreach ($this->routes as $route) {
            $this->cached_output .= $route . "\n";
        }
    }

    public function write(string $filename)
    {
        $this->generateOutput();
        file_put_contents($filename, $this->cached_output);
    }

    public function output()
    {
        $this->generateOutput();
        echo $this->cached_output;
    }

    public function addResourcesFromDir(string $phpNamespace, string $path)
    {
        $dir = new RecursiveDirectoryIterator($path, FilesystemIterator::KEY_AS_PATHNAME, true);
        foreach ($dir as $path => $file) {
            if (substr($path, -4) !== '.php') {
                continue;
            }
            $className = substr(basename($path), 0, -4);
            $fullClassName = $phpNamespace . '\\' . $className;

            $ref = new ReflectionClass($fullClassName);
            $attr = $ref->getAttributes(TypeScript::class);
            $ts = [];
            $code = [];

            if ($attr) {
                foreach ($attr as $att) {
                    foreach ($att->getArguments() as $k => $v) {
                        switch ($k) {
                            case 'name':
                            case 'input':
                            case 'output':
                                throw new TypeScriptGeneratorException("Only code,fields attributes are allowed for resource classes, see: $path");
                            case 'fields':
                                $this->types[$className] = "export interface $className {\n" . $this->cleanupCode($v) . "}";
                                break;
                            case 'code':
                                $code[] = $v;
                                break;
                            default:
                                throw new TypeScriptGeneratorException("Unknown TS attribute: $k");
                        }
                    }
                }
            }

            $this->addCode(...$code);
        }
    }




    /**
     *
     *
     *
     *
     *
     *
     *
     *
     *
     *
     *
     *
     *
     *
     *
     *
     *
     *
     *
     *
     *
     *
     *
     *
     *
     *
     *
     *
     *
     *
     *
     *
     *
     *
     *
     *
     *
     *
     *
     *
     *
     *
     */










    // /**
    //  * Get the processed routes from RouteParser, then parse the docblocks for TS code.
    //  */
    // public function loadRoutes()
    // {
    //     $routes = RouteParser::run();

    //     $len = count($routes);
    //     for ($i = 0; $i < $len; $i++) {
    //         $route = &$routes[$i];

    //         if (preg_match('/ TS:disable\W/', $route['docComment'])) {
    //             unset($routes[$i]);
    //             continue;
    //         }

    //         // Set the default ts:name to equal the endpoint method name
    //         // $route['ts:name'] = $route['method'];
    //         // ALTERNATIVE
    //         // Only include the routes with an included TS:name
    //         if (!preg_match('/ TS:name:/', $route['docComment'])) {
    //             unset($routes[$i]);
    //             continue;
    //         }

    //         // Search through the doccomments for TS: type definitions
    //         preg_match_all(
    //             '/TS:(?<key>[A-Za-z]+):(?<name>[A-Za-z]+)(:((\{\{(?<interface>.*?)\}\})|(?<type>[^\n]+)))?/s',
    //             $route['docComment'],
    //             $matches,
    //             PREG_UNMATCHED_AS_NULL | PREG_SET_ORDER
    //         );


    //         // If none were found, we just continue
    //         if (!count($matches)) {
    //             continue;
    //         }

    //         // Otherwise, loop through each TS: entry in the docblock
    //         foreach ($matches as $m) {
    //             switch ($m['key']) {
    //                     // Override the api call name
    //                 case 'name':
    //                 case 'mutation':
    //                     $route['ts:name'] = $m['name'];
    //                     if ($m['key'] === 'mutation') {
    //                         $route['isMutation'] = true;
    //                     }

    //                     break;
    //                 case 'input':
    //                 case 'output':
    //                     $route['ts:' . $m['key']] = $m['name'];
    //                     // no break
    //                 case 'type':
    //                 case 'query':
    //                     $name = $m['name'];

    //                     // Is this TS type already defined?
    //                     if (isset($this->types[$name])) {
    //                         throw new \Exception("TS type '$name' defined more than once:\n  {$this->types[$name]['filename']}:{$this->types[$name]['line']}\n  {$route['filename']}:{$route['line']}");
    //                     }

    //                     if ($m['key'] === 'query') {
    //                         $route['isQuery'] = true;
    //                     }

    //                     // Is this a type or an interface?
    //                     if (isset($m['type'])) {
    //                         $this->types[$name] = [
    //                             'filename' => $route['filename'],
    //                             'line' => $route['line'],
    //                             'code' => "export type $name = {$m['type']};",
    //                         ];
    //                     } elseif (isset($m['interface'])) {
    //                         $code = $this->cleanupCode($m['interface']);
    //                         $this->types[$name] = [
    //                             'filename' => $route['filename'],
    //                             'line' => $route['line'],
    //                             'code' => "export interface $name {\n$code\n}",
    //                         ];
    //                     }
    //                     break;
    //                 default:
    //                     throw new \Exception("Invalid TS key '{$m['key']}' in {$route['class']}");
    //             }
    //         }
    //     }
    //     $this->routes = $routes;
    // }

    // private function loadApiCalls()
    // {
    //     $out = '';
    //     foreach ($this->routes as $r) {
    //         // Build the JS signature
    //         if (!isset($r['ts:name'])) {
    //             var_dump($r);
    //             exit;
    //         }
    //         $func = "export const {$r['ts:name']} = async (";

    //         // The function parameters
    //         $args = [];
    //         foreach (array_reverse($r['params']) as $p) {
    //             $args[] = "{$p['name']}: {$p['type']}";
    //         }
    //         if (isset($r['ts:input'])) {
    //             $args[] = "data: {$r['ts:input']}";
    //         }
    //         $func .= implode(', ', $args);

    //         // The call to apiRequest
    //         $func .= ') => apiRequest';
    //         // With generics
    //         $func .= '<' . ($r['ts:input'] ?? 'undefined') . ', ' . ($r['ts:output'] ?? 'void') . '>';
    //         // And function arguments
    //         $func .= '(\'' . $r['httpMethod'] . '\', ' . $r['jspath'] . ', ' . (isset($r['ts:input']) ? 'data' : 'undefined') . ');';

    //         $out .= "$func\n";
    //     }
    //     $this->output .= $out;
    // }

    // private function loadQueriesAndMutations()
    // {
    //     $queryOut = '';
    //     $queryKeys = "export const QueryKeys = {\n";
    //     $mutationOut = '';
    //     foreach ($this->routes as $r) {
    //         // Build the JS signature
    //         if (!isset($r['ts:name'])) {
    //             var_dump($r);
    //             exit;
    //         }

    //         $queryArgs = [];
    //         $queryArgsNoTypes = [];
    //         foreach ($r['params'] as $p) {
    //             $queryArgs[] = "{$p['name']}: {$p['type']}";
    //             $queryArgsNoTypes[] = $p['name'];
    //         }

    //         if (isset($r['isQuery']) && $r['isQuery']) {
    //             $tsInput = isset($r['ts:input']) ? $r['ts:input'] : null;
    //             $queryOptionsTypes = ($r['ts:output'] ?? 'void') . ', AxiosError';


    //             $queryOut .= "export const " . ('use' . ucfirst($r['ts:name']) . 'Query') . " = (";
    //             $queryOut .= implode(', ', array_filter([
    //                 ...$queryArgs,
    //                 $tsInput ? "data: {$tsInput}" : null,
    //                 "options?: UseQueryOptions<{$queryOptionsTypes}>"
    //             ]));
    //             $queryOut .= ") =>\n";
    //             $queryOut .= "\tuseQuery<{$queryOptionsTypes}>(\n";

    //             $functParams = implode(', ', array_filter([...$queryArgsNoTypes, $tsInput ? 'data' : null]));

    //             $queryOut .= "\t\tQueryKeys.{$r['ts:name']}({$functParams}),\n";
    //             $queryOut .= "\t\t() => {$r['ts:name']}({$functParams}),\n";
    //             $queryOut .= "\t\toptions\n";
    //             $queryOut .= ");\n";

    //             // Generate the queryKey function
    //             // queryName: (args: [DataType], data: [DataType]) =>
    //             $queryKeys .= "\t{$r['ts:name']}: (" .  implode(', ', array_filter([...$queryArgs, $tsInput ? 'data: ' . $tsInput : null])) . ") => ";
    //             // ['tsName', args, data) =>
    //             $queryKeys .= "[" . implode(', ', array_filter(["'" . $r['ts:name'] . "'", ...$queryArgsNoTypes, $tsInput ? 'data' : null])) . "],\n";
    //         }

    //         if (isset($r['isMutation']) && $r['isMutation']) {
    //             $tsInput = isset($r['ts:input']) ? $r['ts:input'] : '';
    //             $queryArgsNoTypesWithParamsBefore = array_map(function ($arg) {
    //                 return 'params.' . $arg;
    //             }, $queryArgsNoTypes);

    //             $mutationTypes = (isset($r['ts:output']) ? $r['ts:output'] : 'void') . ", AxiosError, ";
    //             $mutationTypes .= "{" . implode('; ', $queryArgs);
    //             if ($tsInput) {
    //                 $mutationTypes .= ($queryArgs ? "; data: " . $tsInput : "data: " . $tsInput);
    //             }
    //             $mutationTypes .= "}";

    //             $mutationOut .= 'export const ' . ('use' . ucfirst($r['ts:name']) . 'Mutation') . " = (";
    //             $mutationOut .= "options?: UseMutationOptions<{$mutationTypes}>) =>";
    //             $mutationOut .= "\n\tuseMutation<{$mutationTypes}>(\n";
    //             $mutationOut .= "\t\t(params) => {$r['ts:name']}(" . implode(', ', $queryArgsNoTypesWithParamsBefore);
    //             if ($tsInput) {
    //                 $mutationOut .= ($queryArgsNoTypesWithParamsBefore ? ", params.data" : 'params.data');
    //             }
    //             $mutationOut .= "),\n";
    //             $mutationOut .= "\t\toptions\n\t);\n";
    //         }
    //     }
    //     $queryKeys .= "};\n";

    //     $this->output .= "\n\n//-----------Queries-----------//\n";
    //     $this->output .= "$queryKeys\n";
    //     $this->output .= "$queryOut\n";
    //     $this->output .= "\n\n//-----------Mutations-----------//\n";
    //     $this->output .= "$mutationOut\n";
    // }
}
